import { __ } from '@wordpress/i18n';
import { InnerBlocks, useBlockProps, InspectorControls, PanelColorSettings } from '@wordpress/block-editor';
import { PanelBody, SelectControl, RangeControl } from '@wordpress/components';
import { useEffect, useState } from '@wordpress/element';
import { useSelect } from '@wordpress/data';
// const ALLOWED_BLOCKS = ['core/column', 'core/cover', 'core/columns', 'core/image', 'core/button', 'core/heading', 'core/paragraph', 'core/group'];

const blockColumns = [
	[
		'core/columns',
		{ className: 'artcloud_fb_columns_main', opacity: 0, overlayColor: 'pale-cyan-blue' },
		[
			[
				'core/column',
				{ className: 'artcloud_fb_column' },
				[
					[
						'core/heading',
						{ lavel: 5, className: 'artcloud_fb_heading', content: 'Teema h3' },
					],
					[
						'core/paragraph',
						{ className: 'artcloud_fb_description', content: 'Excepteur sint occaecat cupidatat non proident, sunt in' },
					],
				],
			],
		],
	],
];

const Edit = ({ attributes, setAttributes }) => {
	const { svgSize, svgHeight, titlePosition, svgColor, selectedSvg, selectedSvgV, selectedSvgA } = attributes;
	const [svgOptions, setSvgOptions] = useState([]);

	useEffect(() => {
		fetch('/wp-json/artcloud/v1/svgs')
			.then(response => response.json())
			.then(data => {
				setSvgOptions(data);
			})
			.catch(error => {
				console.error('Error fetching SVG data:', error);
			});
	}, []);

	const activeThemeName = useSelect((select) => {
		return select('core').getCurrentTheme();
	}, []);

	const defaultSvg = activeThemeName && activeThemeName.name.rendered === 'Vihreat' ? selectedSvgV : selectedSvgA;

	if (!selectedSvg) {
		setAttributes({ selectedSvg: defaultSvg });
	}
	useEffect(() => {
		setAttributes({ selectedSvg: defaultSvg });
	}, [defaultSvg]);

	const handleSelectChange = (newSvg) => {
		setAttributes({ selectedSvg: newSvg });
	};

	const getMaskImageUrl = (selectedOption) => {
		const selectedSvgOption = svgOptions.find((option) => option.value === selectedOption);
		return selectedSvgOption ? `url("data:image/svg+xml,${encodeURIComponent(selectedSvgOption.value)}")` : `url("data:image/svg+xml,${encodeURIComponent(defaultSvg)}")`;
	};
	const onChangeSVGColor = (newColor) => {
		setAttributes({ svgColor: newColor });
	}
	const onChangeSVGSize = (newSize) => {
		setAttributes({ svgSize: newSize });
	}
	const onChangeSVGHeight = (newHeight) => {
		setAttributes({ svgHeight: newHeight });
	}
	const onChangeTitlePosition = (newPosition) => {
		setAttributes({ titlePosition: newPosition });
	}
	return (
		<div {...useBlockProps()}>
			<InspectorControls>
				<PanelBody title={__("Background settings", "artcloud-blocks")} initialOpen={true}>
					<SelectControl
						label={__("Available options", "artcloud-blocks")}
						value={selectedSvg}
						options={svgOptions.map((option) => ({ label: option.label, value: option.value }))}
						onChange={handleSelectChange}
					/>
					<RangeControl
						label={__("SVG Size", "artcloud-blocks")}
						value={svgSize}
						onChange={onChangeSVGSize}
						min={10}
						max={45}
					/>
					<RangeControl
						label={__("SVG Height", "artcloud-blocks")}
						value={svgHeight}
						onChange={onChangeSVGHeight}
						min={10}
						max={45}
					/>
					<RangeControl
						label={__("Title Position", "artcloud-blocks")}
						value={titlePosition}
						onChange={onChangeTitlePosition}
						min={1}
						max={45}
					/>
				</PanelBody>
				<PanelColorSettings
					title={__("Color Settings", "artcloud-blocks")}
					icon="admin-appearance"
					initialOpen
					colorSettings={[
						{
							label: __("SVG Color", "artcloud-blocks"),
							value: svgColor,
							onChange: onChangeSVGColor,
						},
					]}
				></PanelColorSettings>
			</InspectorControls>
			<style>
				{` .wp-block-column.artcloud_fb_column {
                mask-image: ${getMaskImageUrl(selectedSvg)} !important;
                mask-position: center;
				background-color: ${svgColor};
				mask-repeat: no-repeat;
				mask-size: ${svgSize}rem;
				height:${svgSize + 5 > svgHeight ? svgSize + 5 : svgHeight}rem;
				max-width: ${svgSize}rem;
            }`}
			</style>
			<style>
				{` .artcloud_fb_heading{
					top: ${titlePosition}rem;
				}`
				}
			</style>
			<style>
				{` .artcloud_fb_description{
					top: ${titlePosition + 1}rem;
				}
				`}
			</style>
			<InnerBlocks template={blockColumns} />
		</div>
	)
};

export default Edit;
